/*

 Module:   RS1PP.c

 Contains:  RollSCAN-1 Parallel Port and LM9830
			Scanner	Controller Access Functions.

 (c) 2001 Gene Gerety

 Modificaton History

 12-Aug-2001	EPG		Initial
*/

#include <windows.h>
#include <time.h>
#include <conio.h>
#include "RS1PP.h"
#include "RS1_LL.h"

void sleep(DWORD wait);

static BOOL portinit = FALSE;
static WORD portbase = 0;
static WORD portmode = 0;
static WORD ctrlreg = 0;
static WORD statreg = 0;
static WORD eppaddr = 0;
static WORD eppdata = 0;
static WORD ecpecr = 0;

/* Default chip initialization values for RS1Ctlr_Init(); */

static BYTE ChipInit[124][2] =
{
	{0x03,0x00},		/* R03 - Dataport Target - Red Gamma Table */
	{0x04,0x00},		/* R04 - Dataport Address (MSB) - 0x00 */
	{0x05,0x00},		/* R05 - Dataport Address (LSB) - 0x00 */
	{0x06,0x00},		/* R06 - Dataport Data Reg - 0x00 */
	{0x07,0x00},		/* R07 - Command Register - Stop Scan */
	{0x08,0x0A},		/* R08 - MCLK Divider - Div by 2.0 (24.0 MHz) */
	{0x09,0x1D},		/* R09 - HRes/Datamode - div by 6; 8 bit data */
	{0x0a,0x00},		/* R0a - Reserved */
	{0x0b,0x09},		/* R0b - Sensor Config - CIS; No CDS; 600dpi; n=0 */
	{0x0c,0x00},		/* R0c - Sensor controls:polarity - all positive */
	{0x0d,0x27},		/* R0d - Sensor controls:enable */
	{0x0e,0x13},		/* R0e - TR pulse/guardband */
	{0x0f,0x06},		/* R0f - Optical Black Clamp start */
	{0x10,0x17},		/* R10 - Optical Black Clamp end */
	{0x11,0x01},		/* R11 - Reset Pulse Start */
	{0x12,0x02},		/* R12 - Reset Pulse End */
	{0x13,0300},		/* R13 - CP1 Pulse Start */
	{0x14,0x00},		/* R14 - CP1 Pulse End */
	{0x15,0x00},		/* R15 - CP2 Pulse Start */
	{0x16,0x00},		/* R16 - CP2 Pulse End */
	{0x17,0x0b},		/* R17 - Reference Sample Position */
	{0x18,0x01},		/* R18 - Signal Sample Position */
	{0x19,0x00},		/* R19 - CIS Timing Mode */
	{0x1a,0x00},		/* R1a - Reserved */
	{0x1b,0x00},		/* R1b - Reserved */
	{0x1c,0x0f},		/* R1c - Optical Black Pixels Start */
	{0x1d,0x32},		/* R1d - Optical Black Pixels End */
	{0x1e,0x00},		/* R1e - Active Pixels Start MSB */
	{0x1f,0x57},		/* R1f - Active Pixels Start LSB */
	{0x20,0x15},		/* R20 - Line End MSB */
	{0x21,0x27},		/* R21 - Line End LSB */
	{0x22,0x00},		/* R22 - Data Pixels Start MSB */
	{0x23,0x57},		/* R23 - Data Pixels Start LSB */
	{0x24,0x15},		/* R24 - Data Pixels End MSB */
	{0x25,0x1b},		/* R25 - Data Pixels End LSB */
	{0x26,0x01},		/* R26 - Color Mode */
	{0x27,0x00},		/* R27 - Integration Time Adjust (R,G,B) */
	{0x28,0x00},		/* R28 - Reserved */
	{0x29,0x01},		/* R29 - Illumination mode - PWM */
	{0x2a,0x07},		/* R2a - Lamp PWM - MSB */
	{0x2b,0xff},		/* R2b - Lamp PWM - LSB */
	{0x2c,0x00},		/* R2c - Lamp R On - MSB */
	{0x2d,0x01},		/* R2d - Lamp R On - LSB */
	{0x2e,0x03},		/* R2e - Lamp R Off - MSB */
	{0x2f,0xff},		/* R2f - Lamp R Off - LSB */
	{0x30,0x00},		/* R30 - Lamp G On - MSB */
	{0x31,0x00},		/* R31 - Lamp G On - LSB */
	{0x32,0x00},		/* R32 - Lamp G Off - MSB */
	{0x33,0x00},		/* R33 - Lamp G Off - LSB */
	{0x34,0x00},		/* R34 - Lamp B On - MSB */
	{0x35,0x20},		/* R35 - Lamp B On - LSB */
	{0x36,0x00},		/* R36 - Lamp B Off - MSB */
	{0x37,0x30},		/* R37 - Lamp B Off - LSB */
	{0x38,0x00},		/* R38 - Static Offset (Red) */
	{0x39,0x00},		/* R39 - Static Offset (Grn) */
	{0x3a,0x00},		/* R3a - Static Offset (Blu) */
	{0x3b,0x00},		/* R3b - Static Gain (Red) */
	{0x3c,0x00},		/* R3c - Static Gain (Grn) */
	{0x3d,0x00},		/* R3d - Static Gain (Blu) */
	{0x3e,0x1d},		/* R3e - Offset/Gain mode/source control */
	{0x3f,0xaa},		/* R3f - Fixed Offset Coefficient */
	{0x40,0x00},		/* R40 - Fixed Multiplier MSB */
	{0x41,0x55},		/* R41 - Fixed Multiplier LSB */
//	{0x42,0x06},		/* R42 - Parallel Port Settings */
	{0x43,0x1E},		/* R43 - SRAM Settings (Fdx, 256k, max drive current) */
	{0x44,0xff},		/* R44 - Line Skipping n */
	{0x45,0x03},		/* R45 - Stepper Mode (Microstep, motor off) */
	{0x46,0x00},		/* R46 - Scanning Step Size MSB */
	{0x47,0xE2},		/* R47 - Scanning Step Size LSB */
	{0x48,0x00},		/* R48 - Fast Feed Step Size MSB */
	{0x49,0xC6},		/* R49 - Fast Feed Step Size LSB */
	{0x4a,0x00},		/* R4a - Fullsteps to skip MSB */
	{0x4b,0x96},		/* R4b - Fullsteps to skip LSB */
	{0x4c,0x00},		/* R4c - Fullsteps to scan after PS2 - MSB */
	{0x4d,0x00},		/* R4d - Fullsteps to scan after PS2 - LSB */
	{0x4e,0x5c},		/* R4e - Pause scan buffer threshold */
	{0x4f,0x01},		/* R4f - Resume scan buffer threshold */
	{0x50,0x28},		/* R50 - Fullsteps to reverse when buffer full */
	{0x51,0x3f},		/* R51 - Acceleration Profile */
	{0x52,0x07},		/* R52 - DPD - MSB */
	{0x53,0x10},		/* R53 - DPD - LSB */
	{0x54,0x1b},		/* R54 - Lines to process/discard for pause/resume */
	{0x55,0x13},		/* R55 - Kickstart, Hold Current timeout */
	{0x56,0x08},		/* R56 - Stepper PWM freq. */
	{0x57,0x17},		/* R57 - Stepper PWM duty cycle */
	{0x58,0x3b},		/* R58 - Paper Sensor settings */
	{0x59,0xff},		/* R59 - Misc I/O settings */
	{0x5a,0x01},		/* R5a - Line skipping m */
	{0x5b,0x00},		/* R5b - Reserved */
	{0x5c,0x00},		/* R5c - ADC Output Code MSB */
	{0x5d,0x00},		/* R5d - ADC Output Code MSB */
	{0x5e,0x00},		/* R5e - Test settings */
	{0x5f,0x00},		/* R5f - Reserved */
	{0x60,0x00},		/* R60 - Reserved */
	{0x61,0x00},		/* R61 - Reserved */
	{0x62,0x00},		/* R62 - Reserved */
	{0x63,0x00},		/* R63 - Reserved */
	{0x64,0x00},		/* R64 - Reserved */
	{0x65,0x00},		/* R65 - Reserved */
	{0x66,0x00},		/* R66 - Reserved */
	{0x67,0x00},		/* R67 - Reserved */
	{0x68,0x00},		/* R68 - Reserved */
	{0x69,0x00},		/* R69 - Reserved */
	{0x6a,0x00},		/* R6a - Reserved */
	{0x6b,0x00},		/* R6b - Reserved */
	{0x6c,0x00},		/* R6c - Reserved */
	{0x6d,0x00},		/* R6d - Reserved */
	{0x6e,0x00},		/* R6e - Reserved */
	{0x6f,0x00},		/* R6f - Reserved */
	{0x70,0x70},		/* Parallel Port Noise Filter */
	{0x71,0x00},		/* R71 - Reserved */
	{0x72,0x00},		/* R72 - Reserved */
	{0x73,0x00},		/* R73 - Reserved */
	{0x74,0x00},		/* R74 - Reserved */
	{0x75,0x00},		/* R75 - Reserved */
	{0x76,0x00},		/* R76 - Reserved */
	{0x77,0x00},		/* R77 - Reserved */
	{0x78,0x00},		/* R78 - Reserved */
	{0x79,0x00},		/* R79 - Reserved */
	{0x7a,0x00},		/* R7a - Reserved */
	{0x7b,0x00},		/* R7b - Reserved */
	{0x7c,0x00},		/* R7c - Reserved */
	{0x7d,0x00},		/* R7d - Reserved */
	{0x7e,0x00},		/* R7e - Reserved */
	{0x7f,0x00}			/* R7f - Reserved */
};
BOOL RS1PP_SetPortBase(WORD baseaddr) {

	if ((DWORD)baseaddr > 65532)
		return FALSE;
	else if ((DWORD)baseaddr & 0x03)
		return FALSE;
	portbase = (WORD)((WORD)baseaddr & (WORD)0xFFFC);
	ctrlreg = (WORD)(portbase + PP_CREGOFS);
	statreg = (WORD)(portbase + PP_SREGOFS);
	eppaddr = (WORD)(portbase + PP_EPPADDR);
	eppdata = (WORD)(portbase + PP_EPPDATA);
	ecpecr = (WORD)(portbase + PP_ECPECR);
	portinit = FALSE;
	return TRUE;
}


WORD RS1PP_GetPortBase(void) {
	return portbase;
}

BOOL PortInitCheck(void) {
	return portinit;
}

void RS1PP_SetPortMode(BYTE mode)
{
	BYTE regdat;

	portmode = mode;
	// setup the port according to the selected mode
	switch (portmode)
	{
	case PMODE_EPP:
	case PMODE_EPP16:
	case PMODE_EPP32:
	case PMODE_SPP:
	case PMODE_PS2:
		// make sure CREG bits are properly set up
		PORTOUT(ctrlreg,0x04);
		portinit = TRUE;
		break;
	case PMODE_ECPasPS2:
		PORTIN(ecpecr,regdat)
		regdat &= (BYTE)0x1F;
		regdat |= (BYTE)0x20;
		PORTOUT(ecpecr,regdat);
		portinit = TRUE;
		break;
	case PMODE_ECPasEPP:
	case PMODE_ECPasEPP16:
	case PMODE_ECPasEPP32:
		PORTIN(ecpecr,regdat)
		regdat &= (BYTE)0x1F;
		regdat |= (BYTE)0x80;
		PORTOUT(ecpecr,regdat);
		portinit = TRUE;
		break;
	default:
		portinit = FALSE;
	}
}

WORD RS1PP_GetPortMode(void)
{
	return portmode;
}


/*
  Resets the LM9830 as follows:

	1)	Set nStrobe, AutoLF, nInit, and nStrobe to zero
		at the connector
	2)	Wait at least 50 usec
	3)	Set nStrobe, AutoLF, nInit, and nStrobe to one
		at the connector
*/

void RS1Ctlr_Reset(void) {

	if (portbase == 0)
		return;
	PORTOUT(ctrlreg,0x0b)
	sleep(1);
	PORTOUT(ctrlreg,0x00)
	sleep(1);
	PORTOUT(ctrlreg,0x04)
	sleep(1);
}

/*
	Wake up the LM9830 as follows:

	1)	Set nStrobe high (Ctl Reg bit 0 = 0)
		all others static
	2)  Write sequence of 99, 66, cc and 33 to the port
		waiting at least 1 clock tick between writes
*/

void RS1Ctlr_Wake(void) {
	int i;
	BYTE patn[6] = {0x00, 0x99, 0x66, 0xcc, 0x33,0x00};
	BYTE pdata;

	if (portbase == 0)
		return;
	PORTOUT(ctrlreg,0x04)			 /* static control pattern */
	sleep(1);
	for (i=0; i<6; i++) {
		pdata = patn[i];
		PORTOUT(portbase, pdata)
		sleep(1);
	}
}

/*
	"Tranquilize" the LM9830, as follows:

	1) All outgoing control lines inactive (Ctl reg d0-3 = 0x04)
	2) Init Active (Ctl reg d0-3 = 0x00)
	3) Init Inactive

	Waiting one clock tick after each step
*/

void RS1Ctlr_Silence(void)
{

	if (portbase == 0)
		return;
	PORTOUT(ctrlreg, 0x04)
	sleep(1);
	PORTOUT(ctrlreg, 0x00)
	sleep(1);
	PORTOUT(ctrlreg, 0x04)
	sleep(1);
}

/*
	"init" the LM9830 by:

	1) Resetting (see RS1Ctlr_Reset)
	2) Waking (see RS1Ctlr_Wake)
	3) Make sure "Select" status is consistent with LM9830 presence
	4) Test a critical register

  Returns FALSE if 9830 not detected/unresponsive or if
	port not initialized
  Returns TRUE if successful
*/

BOOL RS1Ctlr_Init(void)
{
	int i;
	BYTE dat;

	if (!portinit)
		return FALSE;

	/* reset the LM9830 */
	
	RS1Ctlr_Reset();

	/* wake the LM9830 */

	RS1Ctlr_Wake();

	/* Check the "Select" status -- should be zero */

	PORTIN(statreg,dat);
	if (dat & 0x10)
		return FALSE;		/* LM9830 apparently not present */
	
	/* If one of the EPP modes, make sure EPP timeout bit is clear */

	if ((portmode == PMODE_EPP)||(portmode == PMODE_EPP16)||(portmode == PMODE_EPP32)
		||(portmode == PMODE_ECPasEPP)||(portmode == PMODE_ECPasEPP16)
		||(portmode == PMODE_ECPasEPP32))
	{
		PORTOUT(statreg,0x00)
		PORTOUT(statreg,0x01)
		PORTIN(statreg,dat)
	}

	/* Set parallel port mode to nibble (SPP only) or byte (all others) */

	if (portmode == PMODE_SPP) {
		RS1Ctlr_WriteDataToReg((BYTE)0x42, (BYTE)0x07);
	} else {
		RS1Ctlr_WriteDataToReg((BYTE)0x42, (BYTE)0x06);
	}

	/* write register 8 to set div by 2 MCLK */

	RS1Ctlr_WriteDataToReg((BYTE)0x08,(BYTE)0x04);

	/* now exercise the fixed coefficient registers 3F-41 */

	RS1Ctlr_WriteDataToReg((BYTE)0x3f,(BYTE)0xaa);
	RS1Ctlr_WriteDataToReg((BYTE)0x40,(BYTE)0x00);
	RS1Ctlr_WriteDataToReg((BYTE)0x41,(BYTE)0x55);

	RS1Ctlr_ReadDataFromReg((BYTE)0x3f,&dat);
	if (dat != (BYTE)0xaa)
		return FALSE;
	RS1Ctlr_ReadDataFromReg((BYTE)0x40,&dat);
	if (dat != (BYTE)0x00)
		return FALSE;
	RS1Ctlr_ReadDataFromReg((BYTE)0x41,&dat);
	if (dat != (BYTE)0x55)
		return FALSE;

	/* Now copy chip initialization parameters to LM9830 */
	for (i=0; i<124; i++){
		if (!RS1Ctlr_WriteDataToReg(ChipInit[i][0],ChipInit[i][1]))
			return FALSE;
	}
	return TRUE;
}


/*
	BOOL RS1Ctlr_RamTest()

	Tests the SRAM on the LM9830

	Operates as follows:

		1) Initializes the LM9830
		2) Set SRAM size to 256K, current drive to max
			- 0x1E -> reg 0x43
		3) Set scanner res to 600 dpi (max table size)
			- 0x09 -> reg 0x0B
		4) Set 3-channel color mode
			- 0x00 -> reg 0x26
		5) Write address + 1 to red gamma table
		6) Write address + 2 to grn gamma table
		7) Write address + 3 to blu gamma table
		8) Write address + 5 to red offset/gain
		9) Write address + 7 to grn offset/gain
		10) Write address + 11 to blu offset/gain
		11) Readback and compare gamma and offset/gain

	Any error returns FALSE
	Success returns TRUE

*/ 

BOOL RS1Ctlr_RamTest(void) {
	DWORD i,j,k,l;

	if (!RS1Ctlr_Init()) return FALSE;

	RS1Ctlr_WriteDataToReg((BYTE)0x43,(BYTE)0x1E);
	RS1Ctlr_WriteDataToReg((BYTE)0x0B,(BYTE)0x09);
	RS1Ctlr_WriteDataToReg((BYTE)0x26,(BYTE)0x00);

	/* Write gamma tables */

	for (i=0; i<3; i++) {
		/* Select gamma table */
		RS1Ctlr_WriteDataToReg((BYTE)0x03,(BYTE)(i*2));
		/* Starting address = 0 */
		RS1Ctlr_WriteDataToReg((BYTE)0x04,(BYTE)0x00);
		RS1Ctlr_WriteDataToReg((BYTE)0x05,(BYTE)0x00);
		RS1Ctlr_WriteRegAddr((BYTE)0x06);
		for (j=0; j<1024; j++) {
			RS1Ctlr_WriteData((BYTE)(j+i));
		}
	}

	/* Write gain and offset tables */

	k = 5;
	for (i=0; i<3; i++) {
		/* Select gain/offset table */
		RS1Ctlr_WriteDataToReg((BYTE)0x03,(BYTE)((i*2) + 1));
		/* Starting address = 0 */
		RS1Ctlr_WriteDataToReg((BYTE)0x04,(BYTE)0x00);
		RS1Ctlr_WriteDataToReg((BYTE)0x05,(BYTE)0x00);
		RS1Ctlr_WriteRegAddr((BYTE)0x06);
		k += 2*i;	/* steps through 5,7,11 */
		for (j=0; j<5460; j++) {
			RS1Ctlr_WriteData((BYTE)((j+k)>>8));
			RS1Ctlr_WriteData((BYTE)((j+k)&0xFF));
		}
	}

	/* Now readback and compare, first */

	/* Read gamma tables */

	for (i=0; i<3; i++) {
		BYTE kk;
		/* Select gamma table */
		RS1Ctlr_WriteDataToReg((BYTE)0x03,(BYTE)(i*2));
		/* Starting address = 0 */
		RS1Ctlr_WriteDataToReg((BYTE)0x04,(BYTE)0x20);
		RS1Ctlr_WriteDataToReg((BYTE)0x05,(BYTE)0x00);
		RS1Ctlr_WriteRegAddr((BYTE)0x06);
		for (j=0; j<1024; j++) {
			RS1Ctlr_ReadData(&kk);
			if (kk != (BYTE)(j+i))
				return FALSE;
		}
	}
	
	/* Read gain and offset tables */

	k = 5;
	for (i=0; i<3; i++) {
		BYTE kk,ll;
		/* Select gain/offset table */
		RS1Ctlr_WriteDataToReg((BYTE)0x03,(BYTE)((i*2) + 1));
		/* Starting address = 0 */
		RS1Ctlr_WriteDataToReg((BYTE)0x04,(BYTE)0x20);
		RS1Ctlr_WriteDataToReg((BYTE)0x05,(BYTE)0x00);
		RS1Ctlr_WriteRegAddr((BYTE)0x06);
		k += 2*i;	/* steps through offsets of 5,7,11 */
		for (j=0; j<5460; j++) {
			RS1Ctlr_ReadData(&kk);
			RS1Ctlr_ReadData(&ll);
			l = kk*256 + ll;
			if (l != (j+k))
				return FALSE;
		}
	}

	return TRUE;

}


/*
	Write register address to LM9830
*/

BOOL RS1Ctlr_EppWAddr(BYTE Addr)
{
	BYTE pdata;

	if (portbase == 0)
		return FALSE;
	PORTOUT(eppaddr,Addr)
	PORTIN(statreg,pdata)
	if ((PP_EPPTIMEOUT&pdata)!=0)
		return FALSE;
	else
		return TRUE;
}

BOOL RS1Ctlr_SppWAddr(BYTE Addr)
{
	SPP_WADDR(Addr , RS1SWA)
	return TRUE;
}


/*
	Write data to LM9830
*/

BOOL RS1Ctlr_EppWData(BYTE data)
{
	BYTE pdata;

	if (portbase == 0)
		return FALSE;
	PORTOUT(eppdata,data)
	PORTIN(statreg,pdata)
	if ((PP_EPPTIMEOUT&pdata)!=0)
		return FALSE;
	else
		return TRUE;
}

BOOL RS1Ctlr_SppWData(BYTE data)
{
	SPP_WDATA(data , RS1SWD)
	return TRUE;
}


/*
	Write data to LM9830 (addr write + data write)
*/

BOOL RS1Ctlr_SppWDataReg(BYTE regadr, BYTE data)
{
	SPP_WADDR(regadr, RS1CSRDRa)
	SPP_WDATA(data, RS1CSRDRb)
	return TRUE;
}

BOOL RS1Ctlr_EppWDataReg(BYTE regadr, BYTE data)
{
	EPP_WADDR(regadr)
	EPP_WDATA(data)
	return TRUE;
}

/*
	Read data from LM9830
*/

BOOL RS1Ctlr_EppRData(BYTE *data)
{
	BYTE pdata,qdata;

	if (portbase == 0)
		return FALSE;
	PORTIN(eppdata,qdata)
	PORTIN(statreg,pdata)
	if ((PP_EPPTIMEOUT&pdata)!=0)
		return FALSE;
	*data = qdata;
	return TRUE;
}

BOOL RS1Ctlr_PS2RData(BYTE *data)
{
	BYTE qdata;

	PS2_RDATA(qdata, RS1PRD)
	*data = qdata;
	return TRUE;
}

BOOL RS1Ctlr_SppRData(BYTE *data)
{
	BYTE qdata;

	SPP_RDATA(qdata, RS1SRD)
	*data = qdata;
	return TRUE;
}

BOOL RS1Ctlr_SppRDataReg(BYTE regadr, BYTE *data)
{
	BYTE pdata;

	SPP_WADDR(regadr, RS1CSRDRa)
	SPP_RDATA(pdata, RS1CSRDRb)
	*data = pdata;
	return TRUE;
}

BOOL RS1Ctlr_PS2RDataReg(BYTE regadr, BYTE *data)
{
	BYTE pdata;

	SPP_WADDR(regadr, RS1CPRDRa)
	PS2_RDATA(pdata, RS1CPRDRb)
	*data = pdata;
	return TRUE;
}


BOOL RS1Ctlr_EppRDataReg(BYTE regadr, BYTE *data)
{
	BYTE pdata;

	EPP_WADDR(regadr)
	EPP_RDATA(pdata)
	*data = pdata;
	return TRUE;
}


BOOL RS1Ctlr_WriteRegAddr(BYTE regadr)
{
	if (portbase == 0)
		return FALSE;

	switch (portmode) {
	case PMODE_EPP:
	case PMODE_EPP16:
	case PMODE_EPP32:
	case PMODE_ECPasEPP:
	case PMODE_ECPasEPP16:
	case PMODE_ECPasEPP32:
		EPP_WADDR(regadr)
		break;
	default:
		SPP_WADDR(regadr,RS1CWA)
		break;
	}
	return TRUE;
}


BOOL RS1Ctlr_WriteData(BYTE data)
{
	if (portbase == 0)
		return FALSE;

	switch (portmode) {
	case PMODE_EPP:
	case PMODE_EPP16:
	case PMODE_EPP32:
	case PMODE_ECPasEPP:
	case PMODE_ECPasEPP16:
	case PMODE_ECPasEPP32:
		EPP_WDATA(data)
		break;
	default:
		SPP_WDATA(data,RS1CWD)
		break;
	}
	return TRUE;
}


BOOL RS1Ctlr_WriteDataToReg(BYTE regadr, BYTE data)
{
	if (portbase == 0)
		return FALSE;

	switch (portmode) {
	case PMODE_EPP:
	case PMODE_EPP16:
	case PMODE_EPP32:
	case PMODE_ECPasEPP:
	case PMODE_ECPasEPP16:
	case PMODE_ECPasEPP32:
		EPP_WADDR(regadr)
		EPP_WDATA(data)
		break;
	default:
		SPP_WADDR(regadr,RS1CWDRa)
		SPP_WDATA(data,RS1CWDRb)
		break;
	}
	return TRUE;
}


BOOL RS1Ctlr_ReadData(BYTE *data)
{
	BYTE pdata;

	if (portbase == 0)
		return FALSE;

	switch (portmode) {
	case PMODE_EPP:
	case PMODE_EPP16:
	case PMODE_EPP32:
	case PMODE_ECPasEPP:
	case PMODE_ECPasEPP16:
	case PMODE_ECPasEPP32:
		EPP_RDATA(pdata)
		*data = pdata;
		break;
	case PMODE_PS2:
	case PMODE_ECPasPS2:
		PS2_RDATA(pdata,RS1CRDa);
		*data = pdata;
		break;
	default:
		SPP_RDATA(pdata,RS1CRDb);
		*data = pdata;
		break;
	}
	return TRUE;
}


BOOL RS1Ctlr_ReadDataFromReg(BYTE regadr, BYTE *data)
{
	BYTE pdata;

	if (portbase == 0)
		return FALSE;

	switch (portmode) {
	case PMODE_EPP:
	case PMODE_EPP16:
	case PMODE_EPP32:
	case PMODE_ECPasEPP:
	case PMODE_ECPasEPP16:
	case PMODE_ECPasEPP32:
		EPP_WADDR(regadr)
		EPP_RDATA(pdata)
		*data = pdata;
		break;
	case PMODE_PS2:
	case PMODE_ECPasPS2:
		SPP_WADDR(regadr,RS1CRDRa)
		PS2_RDATA(pdata,RS1CRDRb);
		*data = pdata;
		break;
	default:
		SPP_WADDR(regadr,RS1CRDRc)
		SPP_RDATA(pdata,RS1CRDRd);
		*data = pdata;
		break;
	}
	return TRUE;
}



/*

  Write a list of data values to a corresponding list of
  LM9830 registers.

  "n" is the number of reg/data pairs
  "reglist" is a BYTE array containing the list of register addresses
  "regdata" is a BYTE array containing the list of data values

  returns TRUE if successful, FALSE otherwise.
*/

BOOL RS1Ctlr_WriteRegs(DWORD n, BYTE *reglist, BYTE *regdata)
{
	DWORD i;
	BYTE reg,data;

	switch (portmode) {
	case PMODE_EPP:
	case PMODE_EPP16:
	case PMODE_EPP32:
	case PMODE_ECPasEPP:
	case PMODE_ECPasEPP16:
	case PMODE_ECPasEPP32:
		{
			for (i=0; i<n; i++) {
				reg=reglist[i];
				data=regdata[i];
				EPP_WADDR(reg)
				EPP_WDATA(data)
			}
		}
		break;
	default:
		{
			for (i=0; i<n; i++) {
				reg=reglist[i];
				data=regdata[i];
				SPP_WADDR(reg,RS1CWRa)
				SPP_WDATA(data,RS1CWRb)
			}
		}
		break;
	}
	return TRUE;
}

/*

  Read a list of data values from a corresponding list of
  LM9830 registers.

  "n" is the number of reg/data pairs
  "reglist" is a BYTE array containing the list of register addresses
  "regdata" is a BYTE array to receive the data values

  returns TRUE if successful, FALSE otherwise.
*/
BOOL RS1Ctlr_ReadRegs(DWORD n, BYTE *reglist, BYTE *regdata)
{
	DWORD i;
	BYTE reg,data;

	switch (portmode) {
	case PMODE_EPP:
	case PMODE_EPP16:
	case PMODE_EPP32:
	case PMODE_ECPasEPP:
	case PMODE_ECPasEPP16:
	case PMODE_ECPasEPP32:
		{
			for (i=0; i<n; i++) {
				reg = reglist[i];
				EPP_WADDR(reg)
				EPP_RDATA(data)
				regdata[i] = data;
			}
		}
		break;
	case PMODE_PS2:
	case PMODE_ECPasPS2:
		{
			for (i=0; i<n; i++) {
				reg = reglist[i];
				SPP_WADDR(reg,RS1CRRa)
				PS2_RDATA(data,RS1CRRb)
				regdata[i] = data;
			}
		}
		break;
	default:
		{
			for (i=0; i<n; i++) {
				reg=reglist[i];
				SPP_WADDR(reg,RS1CRRc)
				SPP_RDATA(data,RS1CRRd)
				regdata[i] = data;
			}
		}
		break;
	}
	return TRUE;
}

/*
	Write an n-byte block of data to an LM9830 register

	"n" is number of bytes
	"regadr" is the LM9830 register number
	"data" is a BYTE array, "n" bytes long, of data to be written
*/

BOOL RS1Ctlr_WriteBlockData(DWORD n, BYTE regadr, BYTE *data)
{
	DWORD i;
	BYTE dat;

	if (portbase == 0)
		return FALSE;

	switch (portmode) {
	case PMODE_EPP:
	case PMODE_EPP16:
	case PMODE_EPP32:
	case PMODE_ECPasEPP:
	case PMODE_ECPasEPP16:
	case PMODE_ECPasEPP32:
		{
			EPP_WADDR(regadr)
			PORTOUT_STREAM(eppdata,n,data)
		}
		break;
	default:	/* SPP's and PS2's */
		{
			SPP_WADDR(regadr,RS1CWBa)
			for (i=0; i<n; i++) {
				dat = data[i];
				SPP_WDATA(dat,RS1CWBb)
			}
		}
		break;
	}
	return TRUE;
}

/*
	Read an n-byte block of data from an LM9830 register

	"n" is number of bytes
	"regadr" is the LM9830 register number
	"data" is a BYTE array, "n" bytes long, to receive the data
*/

BOOL RS1Ctlr_ReadBlockData(DWORD n, BYTE regadr, BYTE *data)
{
	DWORD i;
	BYTE *p;
	BYTE pdata;

	if (portbase == 0)
		return FALSE;
	
	switch (portmode) {
	case PMODE_EPP:
	case PMODE_EPP16:
	case PMODE_EPP32:
	case PMODE_ECPasEPP:
	case PMODE_ECPasEPP16:
	case PMODE_ECPasEPP32:
		{
			EPP_WADDR(regadr)
			PORTIN_STREAM(eppdata,n,data)
		}
		break;
	case PMODE_PS2:
	case PMODE_ECPasPS2:
		{
			SPP_WADDR(regadr,RS1CRBa)
			p = data;
			for (i=0; i<n; i++) {
				PS2_RDATA(pdata,RS1CRBb)
				*p++ = pdata;
			}
		}
		break;
	default:
		{
			SPP_WADDR(regadr,RS1CRBc)
			p = data;
			for (i=0; i<n; i++) {
				SPP_RDATA(pdata,RS1CRBd)
				*p++ = pdata;
			}
		}
		break;
	}
	return TRUE;
}

/* Pauses for a specified number of clock ticks. */
void sleep( DWORD wait )
{
   DWORD goal;
   goal = wait + GetTickCount();
   while( goal > GetTickCount() )
      ;
}

