/*

 Module:   RS1PortTest.c

 Contains:  RollSCAN-1 Parallel Port Enumeration and Test Functions

 (c) 2001 Gene Gerety

 Modificaton History

 12-Aug-2001	EPG		Initial
*/

#include <windows.h>
#include <time.h>
#include <conio.h>
#include "RS1PP.h"
#include "RS1_LL.h"
#include <tlhelp32.h>

static int selected_port=-1;
static int selected_mode=0;
static int nports=0;
static int porttype[4];
static WORD portbase[4];

/* Local Function Prototypes */

BOOL TestForEcp(WORD base);
BOOL TestForEpp(WORD base);
BOOL TestForSpp(WORD base);
BOOL TestForPS2(WORD base);
int GetEcpMode(base);

/*
	RS1_EnumeratePorts

	This function determines the number of parallel ports installed and
	their base addresses.

	Function Prototype:

	int RS1_EnumeratePorts(WORD *base);

	where:
		"base" is a WORD array (at least three WORDs long for receiving
				the base addresses of any installed ports

		returns number of ports detected.  0 if no ports detected

 */		// Take inventory of available parallel ports (read PC BIOS config)

int RS1_EnumeratePorts(WORD *paddr, int *ptype) {
	WORD mbuf[4];
	int i;

	Toolhelp32ReadProcessMemory(0,(LPCVOID)0x408,(LPVOID)mbuf,8,NULL);

	nports = 0;
	selected_port = -1;
	for (i=0; i<3; i++) {
		if (mbuf[i]!=0) {
			portbase[nports] = mbuf[i];
			if (paddr != NULL)
				paddr[i] = mbuf[i];
			porttype[nports] = RS1_FindPortType(mbuf[i]);
			if (ptype != NULL)
				ptype[i] = porttype[nports];
			nports++;
		}
	}
	return nports;
}

/*
	RS1_FindPortType

	This function determines the type of parallel port (if any) located at
	the base I/O address given by "port".

	Function Prototype:

	int RS1_FindPortType(WORD port);

	where:
		"port" is a WORD parameter giving the base address of the port to be
		examined

		returns port type detected as follows:

		0 = no port
		1 = SPP
		2 = PS/2 (Bidirectional)
		3 = EPP
		1xx (hex) = ECP, where 'xx' indicates current ECP mode capabilities

 */		// Take inventory of available parallel ports (read PC BIOS config)

int RS1_FindPortType(WORD port) {
	int PortType = RS1_NOPORT;

	if (TestForEcp(port)) {
		PortType = RS1_ECP + GetEcpMode(port);
	} else if ((port != 0x3BC) && (TestForEpp(port))) {
		PortType = RS1_EPP;
	} else if (TestForSpp(port)) {
		if (TestForPS2(port)) {
			PortType = RS1_PS2;
		} else {
			PortType = RS1_SPP;
		}
	}
	return PortType;
}

/*
	 RS1PP_SelectPort	

	The function is called to select a specific parallel port.
	RS1_EnumeratePorts must already have been called and nports
	must be nonzero.

	If the selected port is not PS/2 or EPP capable, the function
	fails (return = FALSE).  If the return value is true, then the
	function has succeeded and the requested port selection is made.

	RollSCAN-1 functions that interact with parallel ports cannot be
	used until this function has been successfully called.

	NOTE:  ports are numbered starting with 0.  The highest numbered
	port is nports-1.

*/
BOOL RS1PP_SelectPort(int portnum) {
	if ((portnum <0)||(portnum>=nports))
		return FALSE;
	if (!RS1PP_SetPortBase(portbase[portnum]))
		return FALSE;
	switch (porttype[portnum]>>8) {
	case 0:	/* unknown */
	case 1: /* SPP */
		return FALSE;

	case 2:	/* PS/2 (bidirectional */
		selected_mode = 1;	/* indicates bidirectional mode */
		break;
	case 3:	/* EPP */
		selected_mode = 2;
		break;
	case 4:	/* ECP */
		if (porttype[portnum]&0x10)
			selected_mode=2;
		else
			selected_mode=1;
		break;

	default:
		return FALSE;
	}
	selected_port=portnum;
	return TRUE;
}



/* Local Helper Functions */
BYTE ECP_EcrRead(base) {
	BYTE data;
	WORD ecraddr;

	ecraddr = (WORD)(base+0x402);
	PORTIN(ecraddr,data)
	return data;
}

void ECP_EcrWrite(WORD base, BYTE data) {
	WORD ecraddr;

	ecraddr = (WORD)(base+0x402);
	PORTOUT(ecraddr,data)
}

BYTE ControlPortRead(WORD base) {
	BYTE data;
	WORD ctrladdr;

	ctrladdr = (WORD)(base+0x02);
	PORTIN(ctrladdr,data)
	data ^= 0x0B;
	return data;
}

void ControlPortWrite(WORD base, BYTE data) {
	WORD ctrlport;

	data ^= 0x0b;
	ctrlport = (WORD)(base+0x02);
	PORTOUT(ctrlport,data)
}

BOOL TestForEcp(WORD base) {
	BOOL flag = FALSE;
	BYTE ecrdata, ctrldata, ctrldatasave;

	ecrdata = ECP_EcrRead(base);
	if ((ecrdata&0x03) == 0x01) {
		ctrldatasave = ControlPortRead(base);  // Control Register
		ctrldata = ctrldatasave;
		if (!(ctrldata & 0x02)) {
			ControlPortWrite(base,(BYTE)0x0F);
			ctrldata = ControlPortRead(base);
		}
		if (((ecrdata&2)^(ctrldata&2))) {
			ECP_EcrWrite(base,(BYTE)0x34);
			ctrldata = ECP_EcrRead(base);
			if (ctrldata == 0x35)
				flag = TRUE;
			ECP_EcrWrite(base,ecrdata);
		}
		ControlPortWrite(base,ctrldatasave);
	}
	return flag;
}

int ReadEppTimeoutBit(base) {
	WORD stataddr;
	BYTE data;

	stataddr = (WORD)(base+1);
	PORTIN(stataddr,data)
	PORTOUT(stataddr,0x01)
	PORTOUT(stataddr,0x00)
	PORTIN(stataddr,data)
	return (data&0x01);
}

BOOL TestForEpp(WORD base) {
	BOOL flag = FALSE;
	WORD testaddr;
	BYTE data;

	testaddr = (WORD)(base+0x03);
	PORTOUT(testaddr,0x55)
	ReadEppTimeoutBit(base);
	PORTIN(testaddr,data)
	ReadEppTimeoutBit(base);
	if (data == 0x55) {
		PORTOUT(testaddr,0xAA)
		ReadEppTimeoutBit(base);
		PORTIN(testaddr,data)
		ReadEppTimeoutBit(base);
		if (data == 0xAA) {
			flag = TRUE;
		}
	}
	return flag;
}

BOOL TestForPS2(WORD base) {
	BYTE data,ctrldatasave;


	ctrldatasave = ControlPortRead(base);
	ControlPortWrite(base,0x2F);		// Tri-state output drivers
	PORTOUT(base,0x55)
	PORTIN(base,data)
	if (data != 0x55)
		return TRUE;
	PORTOUT(base,0xAA)
	PORTIN(base,data)
	ControlPortWrite(base,ctrldatasave);
	if (data != 0xAA)
		return TRUE;
	else
		return FALSE;
}


BOOL TestForSpp(WORD base) {	
	BYTE data,ctrldatasave;


	ctrldatasave = ControlPortRead(base);
	ControlPortWrite(base,0x0F);	// Enable output drivers
	PORTOUT(base,0x55)
	PORTIN(base,data)
	if (data == 0x55) {
		PORTOUT(base,0xAA)
		PORTIN(base,data)
		if (data == 0xAA)
			return TRUE;
			ControlPortWrite(base,ctrldatasave);
	}
	ControlPortWrite(base,ctrldatasave);
	return FALSE;
}

/*
	Determine compatibility with various ECP Mode Options
 */


int GetEcpMode(WORD base) {
	int modes;
	WORD ecraddr;
	BYTE oldecr, data;

	ecraddr = (WORD)(base+0x402);
	
	PORTIN(ecraddr,oldecr)			/* Get original ecr contents */
	/* Test whether EPP mode is supported */
	data = (BYTE)((oldecr & 0x1f)|0x80);
	PORTOUT(ecraddr,data)			/* Try setting EPP mode */
	if (TestForEpp(base))
		modes = 0x1F;
	else
		modes = 0x0F;					/* ALL ECP's support the other modes */

	PORTOUT(ecraddr,oldecr)			/* Restore original ecr contents */
	return modes;
}
