/*
	RS1_LL.h

	Low-level hardware access macro definitions for RollSCAN-1

	NOTE:  These will only work on'x86 machines (i.e., Intel
		486/Pentium machines or equiv) under Win95/95osr2/98/98SE/Me

		They do not function under WinNT/2000/XP because the
		O/S blocks direct access to I/O ports by applications

		For NT/2000/XP, protected-mode drive code is required

 */
/*********************************************************************
 *	DIRECT I/O PORT ACCESS MACROS
 *
 *  PORTOUT, PORTIN, PORTOUT_STREAM, PORTIN_STREAM
 *
 *	These macros are similar to the C library inportb/outportb
 *  but are more efficient because they eliminate function call
 *  overhead and require no stack setup.  
 *********************************************************************/

/* PORTOUT

	Outputs BYTE constant or non-array BYTE variable "data"
	to I/O port designated by WORD constant or non-array
	variable "port"

	(WORD and BYTE are "typedef'd" as follows:

	typedef unsigned short WORD;
	typedef unsigned char BYTE;

	Usage examples:
		The two following examples both write the value "0x0F" to
		I/O port 0x278:

	Example 1:  Variable port/data:

	{
		WORD DataPort;
		BYTE Data;
		.
		.
		.
		DataPort = 0x278;
		Data = 0x0F
		PORTOUT(DataPort,Data);
		.
		.
		.
	}

	Example 2:  Constant port/data

	{
		.
		.
		.
		PORTOUT(0x278,0x0F);
		.
		.
		.
	}
*/

#define PORTOUT(port,data) __asm     \
/* Port output */         \
{                         \
   __asm mov al, data     \
   __asm mov dx, port     \
   __asm out dx, al       \
}	// This comment will catch a stray trailing semicolon
/*---------------------------------------------------------*/


/* PORTIN

	Inputs non-array BYTE variable "data"
	from I/O port designated by WORD constant or non-array
	variable "port"

	(WORD and BYTE are "typedef'd" to unsigned 16 and 8 bit data
	types, respectively)

	Usage examples:
		The two following examples both read a byte from
		I/O port 0x278:

	Variable port/data:

	{
		WORD DataPort;
		BYTE Data;
		.
		.
		.
		DataPort = 0x278;
		Data = 0x0F
		PORTIN(DataPort,Data);
		.
		.
		.
	}

	Constant port/variable data

	{
		BYTE Data;
		.
		.
		.
		PORTIN(0x278,Data);
		.
		.
		.
	}
	
*/

#define PORTIN(port,var) __asm     \
/* Port input */         \
{                         \
   __asm mov dx, port     \
   __asm in  al, dx       \
   __asm mov var, al      \
}	// This comment will catch a stray trailing semicolon
/*---------------------------------------------------------*/

/*
	PORTOUT_STREAM

	Outputs a "stream" of data from a user-specified buffer to
	a specific I/O port.  This is particularly useful in copying
	bulk data to an EPP data port after after an EPP address has
	been specified.

	port must be a 16-bit WORD variable or constant
	count must be a a 16-bit unsigned byte count
	buf MUST be of type BYTE *
	if working from an array, copy the address of the
	array into a byte pointer (unsigned char *) first

	Usage example:

	void Function(...,unsigned char *buf,unsigned short bufsize,...)
	{
		.
		.
		.
		PORTOUT_STREAM(0x278,bufsize,buf);
		.
		.
		.
	}
*/

#define PORTOUT_STREAM(port,count,buf) __asm     \
{                         \
   __asm mov dx, port     \
   __asm mov ecx, count   \
   __asm mov esi, buf     \
   __asm rep outsb        \
}	// This comment will catch a stray trailing semicolon
/*---------------------------------------------------------*/

/*
	PORTIN_STREAM

	Inputs a "stream" of data to a user-specified buffer from
	a specific I/O port.  This is particularly useful in copying
	bulk data from an EPP data port after after an EPP address has
	been specified.

	port must be a 16-bit WORD variable or constant
	count must be a a 16-bit unsigned byte count
	buf MUST be of type BYTE *
	if working from an array, copy the address of the
	array into a byte pointer (unsigned char *) first

	Usage example:

	void Function(...,unsigned char *buf,unsigned short bytecount,...)
	{
		.
		.
		.
		PORTIN_STREAM(0x278,bytecount,buf);
		.
		.
		.
	}*/
#define PORTIN_STREAM(port,count,buf) __asm     \
{                           \
   __asm mov dx, port       \
   __asm mov edi, buf       \
   __asm mov ecx, count     \
   __asm rep insb           \
}	// This comment will catch a stray trailing semicolon

/*---------------------------------------------------------*/

/*********************************************************************
 * EPP (Enhanced Parallel Port) macros
 *
 * EPP_WADDR, EPP_WDATA, EPP_RDATA
 *
 * General information:
 *
 * EPP peripherals (like the RollSCAN-1) are accessed as a set of
 * readable and/or writable registers numbered 0-0xFF (0-255 decimal)
 * To access any register, write its address (using EPP_WADDR) then
 * read or write the register using EPP_RDATA or EPP_WDATA.
 *
 * These operations are most efficient when accomplished using the
 * EPP mode of your parallel port, although they can be accomplished
 * by manipulating the right control and status bits of SPP and PS/2
 * parallel ports.
 *
 * The EPP macros are STRONGLY recommended, since the SPP and PS/2 modes
 * are MUCH slower, and are occasionally unreliable (on some parallel
 * ports.
 *
 * One most EPP peripherals (including the RollSCAN-1) it is only
 * necessary to write a register's address once for multiple accesses
 * to the same register.
 *
 * USAGE EXAMPLE:
 *
 * The following code fragment writes 0x0F to EPP reg #0x0A, writes
 * 0xF0 to EPP reg #0x0B then reads two bytes from EPP reg #0
 *
 * Requirements:
 *      statreg:    WORD variable or constant defining the
 *                  address of the EPP Port's status register
 *                  (portbase+1)
 *      eppaddr:    WORD variable or constant defining the
 *                  address of the EPP Port's address register
 *                  (portbase+3)
 *      eppdata     WORD variable or constant defining the
 *                  address of the EPP Port's data register
 *                  (poerbase+4)
 *
 *  (also see notes above describing PORTOUT, PORTIN, etc.)
 *
 *  void Function(...)
 *  {
 *      unsigned char byte1, byte2;
 *      unsigned short portbase,statreg,ctrlreg,eppaddr,eppdata
 *
 *      portbase=0x378;
 *      statreg=0x379;
 *      ctrlreg=0x37a;
 *      eppaddr=0x37b;
 *      eppdata=0x37c;
 *      .
 *      .
 *      .
 *      // Write 0x0F to RollSCAN-1 register 0x0A
 *      EPP_WADDR(0x0A);
 *      EPP_WDATA(0x0F);
 *      // Write 0xF0 to RollSCAN-1 register 0x0B
 *      EPP_WADDR(0x0B);
 *      EPP_WDATA(0xF0);
 *      // Read two bytes from RollSCAN-1 register 0x00
 *      EPP_WADDR(0x00);
 *      EPP_RDATA(byte1);
 *      EPP_RDATA(byte2);
 *      .
 *      .
 *      .
 *      }
 *
 *********************************************************************/

/*
	EPP_WADDR
	Perform an address write to the LM9830 - EPP MODE
	Steps:
		1) Clear EPP busy bit
			- write 0 to status reg
			- write 1 to status reg
			- read status reg
		2) Write EPP address register (base + 3)

*/
#define EPP_WADDR(addr) __asm \
{									\
	__asm		mov dx,statreg		\
	__asm		mov al,0			\
	__asm		out dx,al			\
	__asm		in  al,dx			\
	__asm		mov dx,eppaddr		\
	__asm		mov al,addr			\
	__asm		out dx,al			\
}	// This comment will catch a stray trailing semicolon

/*
	EPP_WDATA
	Perform a data write to the LM9830 - EPP MODE
	Steps:
		1) Clear EPP busy bit
			- write 0 to status reg
			- write 1 to status reg
			- read status reg
		2) Write EPP data register (base + 4)

*/
#define EPP_WDATA(data) __asm \
{									\
	__asm		mov dx,statreg		\
	__asm		mov al,0			\
	__asm		out dx,al			\
	__asm		in  al,dx			\
	__asm		mov dx,eppdata		\
	__asm		mov al,data			\
	__asm		out dx,al			\
}	// This comment will catch a stray trailing semicolon

/*
	EPP_RDATA
	Perform a data read from the LM9830 - EPP MODE
	Steps:
		1) Clear EPP busy bit
			- write 0 to status reg
			- write 1 to status reg
			- read status reg
		2) Read EPP data register (base + 4)

*/
#define EPP_RDATA(data) __asm \
{									\
	__asm		mov dx,statreg		\
	__asm		mov al,0			\
	__asm		out dx,al			\
	__asm		inc al				\
	__asm		out dx,al			\
	__asm		in  al,dx			\
	__asm		mov dx,eppdata		\
	__asm		in  al,dx			\
	__asm		mov data,al			\
}	// This comment will catch a stray trailing semicolon
/*---------------------------------------------------------*/


/*
	SPP_WADDR, SPP_WDATA and SPP_RDATA
	Perform address writes, data writes, and data reads to/from
	the LM9830 using SPP (Standard Parallel Port) controls

	SPP_RDATA
	Performs an 8-bit data read from the LM9830 as a pair of
	4-bit reads using the status register for input.  Bit 3-6
	of the status register are used

	This is the so-called "nibble-mode" operation where data
	reading is accomplished via status register bits - 4 bits at
	a time.  The performance of this mode of operation is very
	poor compared to the performance of the EPP macros.  Use these
	functions ONLY if an EPP or Bidirectional port is unavailable.

	Steps:
		SPP_WDATA:
		1) Controls to idle (ctrlreg == 0x04)
		2) Wait for not busy (statreg bit7 == 0)
		3) output data to datareg
		4) turn on "strobe" (creg=0x05)
		5) turn on "autofeed" (creg=0x07)
		6) look for busy status (statreg bit7 == 1)
		7) turn off "strobe" and "autofeed" (creg=0x04)
		8) wait for not busy
		9) write 00 to data reg

		SPP_WDATA
		1) Controls to idle (ctrlreg == 0x04)
		2) Wait for not busy (statreg bit7 == 0)
		3) output data to datareg
		4) turn on "strobe" (creg=0x05)
		5) turn on "select in" (creg=0x0d)
		6) look for busy status (statreg bit7 == 1)
		7) turn off "strobe" and "select in" (creg=0x04)
		8) wait for not busy
		9) write 00 to data reg (per Par Port spec recommendations)

		SPP_RDATA
		1) Wait for not busy (statreg bit7 == 0)
		2) turn on "autofeed" (creg=0x06)
		3) wait for busy status (statreg bit7 == 1)
		4) input data (ms nibble) from status reg into al
		5) mask and rotate left into bits 4-7, save in ah
		6) turn off "autofeed" (creg=0x04)
		7) wait for not busy (statreg bit7 == 0)
		8) input data (ls nibble) from status reg into al
		9) mask and rotate right into bits 0-3
		10) "or" with ms nibble in ah

  **WARNING**

	1) On some machines, the SPP reads and writes have proven to be
		unreliable.  It is STRONGLY recommended that the EPP functions
		be used wherever possible.

	2) If peripheral is not connected to parallel port, this macro may
		cause the program to "hang", because the "BUSY" status bit will
		be unresponsive

	Requirements:
		ctrlreg:	unsigned short (16 bit) loaded with address of
					parallel port control register (base addr + 2)
		statreg:	unsigned short (16 bit) loaded with address of
					parallel port status register (base addr + 1)
		portbase:	unsigned short (16 bit) loaded with base address
					of parallel port

		"labelbase" is a text label (not a literal -- i.e., no quotes)
		that must be unique to each invocation of the SPP_WADDR, SPP_WDATA
		or SPP_RDATA macro within a source code module (file).

		Usage Example:

		This example illustrates reading and writing from two different
		LM9830 registers.  The numbers used in this example are for
		illustrative purposes only and are not intended to get the LM9830
		to do anything special.

		The example writes a 0x01 to LM9830 register 0x0A, then writes
		0x02 to LM9830 register 0x0B.  It then reads two values from the
		LM9830 Data register - register 00.

		void SPP_Example( ... )
		{
			WORD portbase = 0x278;
			WORD statreg = 0x279;
			WORD ctlreg = 0x27A;
			BYTE data1, data2;
			.
			.
			.
			SPP_WADDR(0x0A,WA1);	// Write the LM9830 register address (0x0A)
			SPP_WDATA(0x01,WD1);	// Write 0x01 to register 0x0A
			SPP_WADDR(0x0B,WA2);	// Write the LM9830 register address (0x0B)
			SPP_WDATA(0x02,WD2);	// Write 0x02 to register 0x0B
			SPP_WADDR(0x00,WA3);	// Write the LM9830 register address (0x00)
			SPP_RDATA(data1,RD1);	// Read first data byte from register 00
			SPP_RDATA(data2,RD2);	// Read second data byte from register 00
			.
			.
			.
		}
*/
#define SPP_WADDR(addr, labelbase) __asm \
{									\
	__asm		mov dx,ctrlreg		\
	__asm		mov	al,0x04			\
	__asm		out	dx,al			\
	__asm		mov dx,statreg		\
	__asm	labelbase##a:			\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jz  labelbase##a	\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jz  labelbase##a	\
	__asm		mov dx,portbase		\
	__asm		mov al,addr			\
	__asm		out dx,al			\
	__asm		mov dx,ctrlreg		\
	__asm		mov al,0x05			\
	__asm		out dx,al			\
	__asm		mov al,0x0d			\
	__asm		out dx,al			\
	__asm		mov dx,statreg		\
	__asm	labelbase##b:			\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jnz labelbase##b	\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jnz labelbase##b	\
	__asm		mov dx,ctrlreg		\
	__asm		mov al,0x04			\
	__asm		out dx,al			\
	__asm		mov dx,statreg		\
	__asm	labelbase##c:			\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jz  labelbase##c	\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jz  labelbase##c	\
	__asm		mov dx,portbase		\
	__asm		xor al,al			\
	__asm		out dx,al			\
}	// This comment will catch a stray trailing semicolon

#define SPP_WDATA(data , labelbase) __asm \
{									\
	__asm		mov dx,ctrlreg		\
	__asm		mov	al,0x04			\
	__asm		out	dx,al			\
	__asm		mov dx,statreg		\
	__asm	labelbase##a:			\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jz  labelbase##a	\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jz  labelbase##a	\
	__asm		mov dx,portbase		\
	__asm		mov al,data			\
	__asm		out dx,al			\
	__asm		mov dx,ctrlreg		\
	__asm		mov al,0x05			\
	__asm		out dx,al			\
	__asm		mov al,0x07			\
	__asm		out dx,al			\
	__asm		mov dx,statreg		\
	__asm	labelbase##b:			\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jnz labelbase##b	\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jnz labelbase##b	\
	__asm		mov dx,ctrlreg		\
	__asm		mov al,0x04			\
	__asm		out dx,al			\
	__asm		mov dx,statreg		\
	__asm	labelbase##c:			\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jz  labelbase##c	\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jz  labelbase##c	\
	__asm		mov dx,portbase		\
	__asm		xor al,al			\
	__asm		out dx,al			\
}	// This comment will catch a stray trailing semicolon

#define SPP_RDATA(data, labelbase) __asm \
{									\
	__asm	labelbase##a:			\
	__asm		mov dx,statreg		\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jz  labelbase##a	\
	__asm		mov dx,ctrlreg		\
	__asm		mov al,0x06			\
	__asm		out dx,al			\
	__asm	labelbase##b:			\
	__asm		mov dx,statreg		\
	__asm		in  al,dx			\
	__asm		bt  ax,0x07			\
	__asm		jc    labelbase##b	\
	__asm		and al,0x78			\
	__asm		rol al,1			\
	__asm		mov ah,al			\
	__asm		mov dx,ctrlreg		\
	__asm		mov al,0x04			\
	__asm		out dx,al			\
	__asm	labelbase##c:			\
	__asm		mov dx,statreg		\
	__asm		in  al,dx			\
	__asm		bt  ax,0x07			\
	__asm		jnc labelbase##c	\
	__asm		and al,0x78			\
	__asm		ror al,0x03			\
	__asm		or  al,ah			\
	__asm		mov data,al			\
}	// This comment will catch a stray trailing semicolon
/*---------------------------------------------------------*/

/*
	PS2_RDATA
	Perform an 8-bit data read from the LM9830 - PS/2 mode
	Steps:
		1) Controls to idle (ctrlreg == 0x04)
		2) Wait for not busy (statreg bit7 == 0)
		3) Tri-state bus (ctrlreg == 0x24)
		4) turn on "autofeed" (creg=0x26)
		5) wait for busy status (statreg bit7 == 1)
		6) input data from datareg
			-- kill some time by rewriting control register
		7) turn off "autofeed" (creg=0x24)
		8) wait for not busy (statreg bit7 == 0)
		8) turn off tri-state (creg=0x04)
		9) 0x00 to data bus

    Note:  PS/2 ports differ from SPP ports only in the way they
        read data.  For address and data writes, use SPP_WADDR 
        and SPP_WDATA.

  **WARNING**

	If peripheral is not connected to parallel port, this macro may
	cause the program to "hang", because the "BUSY" status bit will
	be unresponsive
*/

#define PS2_RDATA(data, labelbase) __asm \
{									\
	__asm		mov dx,ctrlreg		\
	__asm		mov	al,0x04			\
	__asm		out	dx,al			\
	__asm		mov dx,statreg		\
	__asm	labelbase##a:			\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jz  labelbase##a	\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jz  labelbase##a	\
	__asm		mov dx,ctrlreg		\
	__asm		mov al,0x24			\
	__asm		out dx,al			\
	__asm		mov al,0x26			\
	__asm		out dx,al			\
	__asm		mov dx,statreg		\
	__asm	labelbase##b:			\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jnz labelbase##b	\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jnz labelbase##b	\
	__asm		mov dx,portbase		\
	__asm		in  al,dx			\
	__asm		mov data,al			\
	__asm		mov dx,ctrlreg		\
	__asm		in  al,dx			\
	__asm		in  al,dx			\
	__asm		in  al,dx			\
	__asm		mov al,0x26			\
	__asm		out dx,al			\
	__asm		mov al,0x24			\
	__asm		out dx,al			\
	__asm		mov dx,statreg		\
	__asm	labelbase##c:			\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jz  labelbase##c	\
	__asm		in  al,dx			\
	__asm		and al,0x80			\
	__asm		jz  labelbase##c	\
	__asm		mov dx,ctrlreg		\
	__asm		mov al,0x04			\
	__asm		out dx,al			\
	__asm		mov dx,portbase		\
	__asm		xor al,al			\
	__asm		out dx,al			\
}	// This comment will catch a stray trailing semicolon
/*---------------------------------------------------------*/

